<?php
set_time_limit(0);

define('ZIP_NAME', 'prestashop.zip');
define('TARGET_FOLDER', __DIR__.'/');
define('BATCH_SIZE', 500);

$startId = 0;
if (isset($_POST['startId'])) {
  $startId = (int)$_POST['startId'];
}

if (isset($_POST['extract'])) {
  if (!extension_loaded('zip')) {
    die(json_encode([
      'error' => true,
      'message' => 'You must install PHP zip extension first',
    ]));
  }

  $zip = new ZipArchive();
  if ($zip->open(__DIR__.'/'.ZIP_NAME) === true) {

    $numFiles = $zip->numFiles;
    $lastId = $startId + BATCH_SIZE;

    $fileList = array();
    for ($id = $startId; $id < min($numFiles, $lastId); $id++) {
      $currentFile = $zip->getNameIndex($id);
      if (in_array($currentFile, ['/index.php', 'index.php'])) {
        $indexContent = $zip->getFromIndex($id);
        file_put_contents(getcwd().'/index.php.temp', $indexContent);
      } else {
        $fileList[] = $currentFile;
      }
    }

    foreach ($fileList as $currentFile) {
      if ($zip->extractTo(TARGET_FOLDER, $currentFile) === false) {
        die(json_encode([
          'error' => true,
          'message' => 'An error occured during the extraction',
          'file' => $currentFile,
          'status' => $zip->getStatusString(),
          'numFiles' => $numFiles,
          'lastId' => $lastId,
          'files' => $fileList,
        ]));
      }
    }

    $zip->close();

    if ($lastId >= $numFiles) {
      unlink(getcwd().'/index.php');
      unlink(getcwd().'/prestashop.zip');
      rename(getcwd().'/index.php.temp', getcwd().'/index.php');
      sleep(2); // we need to wait the rename creation as the ajax call is asynchronous
    }

    die(json_encode([
      'error' => false,
      'numFiles' => $numFiles,
      'lastId' => $lastId,
    ]));
  }
}

if (isset($_GET['element'])) {
  switch ($_GET['element']) {
    case 'font':
      header('Content-Type: application/font-sfnt');
      echo base64_decode(<<<EOF
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
EOF
);
      break;
    case 'css':
      header('Content-Type: text/css');
      echo base64_decode(<<<EOF
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
EOF
);
      break;
    case 'jquery':
      header('Content-Type: text/javascript');
      echo base64_decode(<<<EOF
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
EOF
);
      break;
    case 'logo':
      header('Content-Type: image/png');
      echo base64_decode(<<<EOF
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
EOF
);
    case 'gif':
      header('Content-Type: image/png');
      echo base64_decode(<<<EOF
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
EOF
);
    break;
  }
  exit;
}

?>
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>PrestaShop installation</title>
  <link rel="stylesheet" type="text/css" href="index.php?element=css">
</head>
<body>
  <img id="logo" src="index.php?element=logo" style="width: 200px;" />
  <div id="content">
    <div>
      <img src="index.php?element=gif" style="width: 480px;" />

      <div id="progressContainer">
        <div class="progressNumber">0 %</div>
        <div class="progress">
          <div class="current">
          </div>
        </div>
      </div>

      <div id="error"></div>
    </div>
  </div>
  <script type="text/javascript" src="index.php?element=jquery"></script>
  <script type="text/javascript">

    function extractFiles(startId) {

      if (typeof startId == 'undefined') {
        startId = 0;
      }

      var request = $.ajax({
        method: "POST",
        url: "index.php",
        data: {
          extract: true,
          startId: startId,
        }
      });

      request.done(function(msg) {
        try {
          msg = JSON.parse(msg);
        }catch(e){
          msg = {
            message: msg
          };
        }

        if (
          msg.fail
          || typeof msg.lastId == 'undefined'
          || typeof msg.numFiles == 'undefined'
        ) {
          $('#error').html('An error has occured : <br />'+ msg.message);
          $('.spinner').remove();
        } else {
          if (msg.lastId > msg.numFiles) {
            location.reload();
          } else {
            $("#progressContainer")
              .find(".current")
              .width((msg.lastId / msg.numFiles * 100)+'%');

            $("#progressContainer")
              .find(".progressNumber")
              .css({left: Math.round((msg.lastId / msg.numFiles * 100))+'%'})
              .html(Math.round((msg.lastId / msg.numFiles * 100))+'%');

            extractFiles(msg.lastId);
          }
        }
      });

      request.fail(function(jqXHR, textStatus, errorThrown) {
        $('#error').html('An error has occured' + textStatus);
        $('.spinner').remove();
      });
    }

    $(function() {
      extractFiles();
    });
  </script>
</body>
</html>
