<?php

/**
 * Copyright since 2007 PrestaShop SA and Contributors
 * PrestaShop is an International Registered Trademark & Property of PrestaShop SA
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    PrestaShop SA and Contributors <contact@prestashop.com>
 * @copyright Since 2007 PrestaShop SA and Contributors
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */

namespace PrestaShop\Module\PrestashopCheckout\PayPal\Order\QueryHandler;

use PrestaShop\Module\PrestashopCheckout\Exception\HttpTimeoutException;
use PrestaShop\Module\PrestashopCheckout\PayPal\Order\Exception\PayPalOrderException;
use PrestaShop\Module\PrestashopCheckout\PayPal\Order\Query\GetPayPalOrderForCheckoutCompletedQuery;
use PrestaShop\Module\PrestashopCheckout\PayPal\Order\Query\GetPayPalOrderForCheckoutCompletedQueryResult;
use PrestaShop\Module\PrestashopCheckout\PaypalOrder;
use Symfony\Component\Cache\Adapter\ChainAdapter;

/**
 * We need to know if the Order Status is APPROVED and in case of Card payment if 3D Secure allow to capture
 */
class GetPayPalOrderForCheckoutCompletedQueryHandler
{
    public function __construct(private ChainAdapter $orderPayPalCache)
    {
    }

    public function __invoke(GetPayPalOrderForCheckoutCompletedQuery $getPayPalOrderQuery)
    {
        $payPalOrderId = $getPayPalOrderQuery->getOrderPayPalId()->getValue();

        /** @var array{id: string, status: string}|array $order */
        $order = $this->orderPayPalCache->getItem($payPalOrderId)->get();

        if (!empty($order) && in_array($order['status'], ['COMPLETED', 'CANCELED'])) {
            return new GetPayPalOrderForCheckoutCompletedQueryResult($order);
        }

        try {
            $orderPayPal = new PaypalOrder($payPalOrderId);
            $orderToStoreInCache = !empty($order) ? array_replace_recursive($order, $orderPayPal->getOrder()) : $orderPayPal->getOrder();
            $cacheItem = $this->orderPayPalCache->getItem($payPalOrderId);
            $cacheItem->set($orderToStoreInCache);
            $this->orderPayPalCache->save($cacheItem);
        } catch (HttpTimeoutException $exception) {
            throw $exception;
        } catch (\Exception $exception) {
            throw new PayPalOrderException(sprintf('Unable to retrieve PayPal Order %s', $payPalOrderId), PayPalOrderException::CANNOT_RETRIEVE_ORDER, $exception);
        }

        if (!$orderPayPal->isLoaded()) {
            throw new PayPalOrderException(sprintf('No data for PayPal Order %s', $payPalOrderId), PayPalOrderException::EMPTY_ORDER_DATA);
        }

        return new GetPayPalOrderForCheckoutCompletedQueryResult($orderPayPal->getOrder());
    }
}
